<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use App\Models\TargetSavingsSchedule;
use App\Services\TargetSavingsService;

class TargetSavingsCommand extends Command
{
    protected $signature = 'app:target-savings-command';
    protected $description = 'Automatically debit pending target savings schedules via cron';

    public function handle(TargetSavingsService $service)
    {
        $runDate = Carbon::today()->toDateString();
        $this->info("Running target debits for {$runDate}");

        $schedules = TargetSavingsSchedule::query()
            ->where('status', 'pending')
            ->whereDate('due_date', '=', $runDate)
            ->orderBy('due_date', 'asc')
            ->get();

        if ($schedules->isEmpty()) {
            $this->info('No pending schedules due today.');
            return self::SUCCESS;
        }

        $processed = 0;

        foreach ($schedules as $sch) {
            try {
                // Double-guard (cheap)
                if ($sch->status !== 'pending') {
                    continue;
                }

                $res = $service->ScheduleCronJob($sch->id, $runDate);
                $status = $res['status'] ?? 'failed';
                $code   = $res['status_code'] ?? '96';
                $msg    = $res['message'] ?? '';
                $this->line("Schedule #{$sch->id} - {$status} [{$code}] - {$msg}");

                $processed++;
            } catch (\Throwable $e) {
                Log::error('Debit schedule failed', [
                    'schedule_id' => $sch->id,
                    'error'       => $e->getMessage(),
                ]);
                $this->error("Schedule #{$sch->id} - failed [96] - {$e->getMessage()}");
            }
        }

        $this->info("Processed: {$processed} schedule(s) due today.");
        return self::SUCCESS;
    }
}
