<?php

namespace App\Http\Controllers;

use Exception;
use App\Services\MatrixPayService;
use Carbon\Carbon;
use App\Models\Loan;
use App\Models\Email;
use App\Models\Saving;
use App\Models\Status;
use App\Models\Setting;
use App\Models\Customer;
use Illuminate\Support\Str;
use App\Models\Exchangerate;
use Illuminate\Http\Request;
use App\Models\Accountofficer;
use App\Models\SavingsProduct;
use Illuminate\Validation\Rule;
use App\Http\Traites\LoanTraite;
use App\Http\Traites\UserTraite;
use App\Http\Traites\AuditTraite;
use App\Http\Traites\SavingTraite;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Maatwebsite\Excel\Facades\Excel;
use App\Notifications\AppNotification;
use App\Exports\CustomersBalanceExport;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Schema;

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Shared\Date as PhpSpreadsheetDate;


class CustomersController extends Controller
{
    use AuditTraite;
    use SavingTraite;
    use LoanTraite;
    use UserTraite;

    public function __construct(
        private MatrixPayService $matrixPay,
    )
    {
        $this->middleware('auth');
    }

    public function manage_customers()
    {
        if (Auth::user()->roles()->first()->name == 'account officer') {

            $acofficer = Accountofficer::select('id')->where('user_id', Auth::user()->id)->first();
            $customers = Customer::select('id', 'last_name', 'first_name', 'account_category', 'ctype', 'business_name', 'acctno', 'refacct', 'account_type', 'username', 'phone', 'gender', 'email', 'reg_date', 'status')
                ->where('accountofficer_id', $acofficer->id)
                ->where('status', '1')
                ->orderBy('created_at', 'DESC')->get();
            // dd($customers);
            return view('customers.manage_customer')->with('customers', $customers);
        } else {

            $cust = Customer::select('id', 'last_name', 'first_name', 'account_category', 'ctype', 'business_name', 'acctno', 'refacct', 'account_type', 'username', 'phone', 'gender', 'email', 'reg_date', 'status')
                ->where('status', '1')->orderBy('created_at', 'DESC')->get();

            return view('customers.manage_customer')->with('customers', $cust);
        }
    }

    public function manage_pending_customers(Request $request)
    {
        $search = $request->input('search');

        if (Auth::user()->roles()->first()->name == 'account officer') {

            $acofficer = Accountofficer::where('user_id', Auth::user()->id)->first();

            $customers = Customer::select(
                'id',
                'last_name',
                'first_name',
                'account_category',
                'ctype',
                'business_name',
                'acctno',
                'refacct',
                'account_type',
                'username',
                'phone',
                'gender',
                'email',
                'reg_date',
                'status'
            )
                ->where('accountofficer_id', $acofficer->id)
                ->where('status', '7')
                ->when($search, function ($query) use ($search) {
                    $query->where(function ($q) use ($search) {
                        $q->where('last_name', 'like', "%{$search}%")
                            ->orWhere('first_name', 'like', "%{$search}%")
                            ->orWhere('business_name', 'like', "%{$search}%")
                            ->orWhere('acctno', 'like', "%{$search}%")
                            ->orWhere('refacct', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%")
                            ->orWhere('email', 'like', "%{$search}%");
                    });
                })
                ->orderBy('created_at', 'DESC')
                ->paginate(20);

            // keep query string (search) on pagination links
            $customers->appends($request->query());

            return view('customers.manage_pending_customer')
                ->with('pdcustomers', $customers);
        } else {

            $pcust = Customer::select(
                'id',
                'last_name',
                'first_name',
                'account_category',
                'acctno',
                'ctype',
                'business_name',
                'refacct',
                'username',
                'account_type',
                'phone',
                'gender',
                'email',
                'reg_date',
                'status'
            )
                ->where('status', '7')
                ->when($search, function ($query) use ($search) {
                    $query->where(function ($q) use ($search) {
                        $q->where('last_name', 'like', "%{$search}%")
                            ->orWhere('first_name', 'like', "%{$search}%")
                            ->orWhere('business_name', 'like', "%{$search}%")
                            ->orWhere('acctno', 'like', "%{$search}%")
                            ->orWhere('refacct', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%")
                            ->orWhere('email', 'like', "%{$search}%");
                    });
                })
                ->orderBy('created_at', 'DESC')
                ->paginate(20);

            $pcust->appends($request->query());

            return view('customers.manage_pending_customer')
                ->with('pdcustomers', $pcust);
        }
    }



    public function manage_closed_customers()
    {
        if (Auth::user()->roles()->first()->name == 'account officer') {

            $acofficer = Accountofficer::where('user_id', Auth::user()->id)->first();
            $customers = Customer::select('id', 'last_name', 'first_name', 'account_category', 'ctype', 'business_name', 'acctno', 'refacct', 'account_type', 'username', 'phone', 'gender', 'email', 'reg_date', 'status')
                ->where('accountofficer_id', $acofficer->id)
                ->where('status', '2')
                ->orderBy('created_at', 'DESC')->get();

            return view('customers.manage_closed_customer')->with('clcustomers', $customers);
        } else {

            $cust = Customer::select('id', 'last_name', 'ctype', 'first_name', 'account_category', 'business_name', 'acctno', 'refacct', 'username', 'account_type', 'phone', 'gender', 'email', 'reg_date', 'status')
                ->where('status', '2')->orderBy('created_at', 'DESC')->get();
            return view('customers.manage_closed_customer')->with('clcustomers', $cust);
        }
    }
    public function manage_restricted_customers()
    {
        if (Auth::user()->roles()->first()->name == 'account officer') {

            $acofficer = Accountofficer::where('user_id', Auth::user()->id)->first();
            $customers = Customer::select('id', 'last_name', 'first_name', 'ctype', 'account_category', 'business_name', 'acctno', 'refacct', 'account_type', 'username', 'phone', 'gender', 'email', 'reg_date', 'status')
                ->where('accountofficer_id', $acofficer->id)
                ->whereBetween('status', ['4', '6'])
                ->orderBy('created_at', 'DESC')->get();

            return view('customers.manage_restricted_customer')->with('clcustomers', $customers);
        } else {

            $rcust = Customer::select('id', 'last_name', 'first_name', 'ctype', 'account_category', 'business_name', 'acctno', 'refacct', 'account_type', 'username', 'phone', 'gender', 'email', 'reg_date', 'status')
                ->whereBetween('status', ['4', '6'])->orderBy('created_at', 'DESC')->get();
            return view('customers.manage_restricted_customer')->with('clcustomers', $rcust);
        }
    }
    public function manage_dom_accounts()
    {
        if (Auth::user()->roles()->first()->name == 'account officer') {

            $acofficer = Accountofficer::where('user_id', Auth::user()->id)->first();
            $customers = Customer::select('id', 'last_name', 'first_name', 'ctype', 'business_name', 'account_category', 'account_type', 'acctno', 'refacct', 'username', 'phone', 'gender', 'email', 'reg_date', 'status')
                ->where('accountofficer_id', $acofficer->id)
                ->where('status', '8')
                ->orderBy('created_at', 'DESC')->get();

            return view('customers.manage_dom_account')->with('clcustomers', $customers);
        } else {

            $dcust = Customer::select('id', 'last_name', 'first_name', 'ctype', 'business_name', 'account_category', 'acctno', 'refacct', 'account_type', 'username', 'phone', 'gender', 'email', 'reg_date', 'status')
                ->where('status', '8')->orderBy('created_at', 'DESC')->get();
            return view('customers.manage_dom_account')->with('clcustomers', $dcust);
        }
    }


    public function view_customer()
    {
        if (request()->filter == true) {

            $customer = Customer::select('id', 'last_name', 'first_name', 'ctype', 'business_name', 'account_category', 'acctno', 'refacct', 'account_type', 'username', 'phone', 'gender', 'email', 'reg_date', 'status')
                ->where('acctno', request()->csdetails)
                ->orWhere('first_name', 'like', '%' . request()->csdetails . '%')
                ->orWhere('last_name', 'like', '%' . request()->csdetails . '%')
                ->orWhere('business_name', 'like', '%' . request()->csdetails . '%')
                ->get();

            return view('customers.view_customer')->with('customers', $customer);
        } else {

            return view('customers.view_customer');
        }
    }

    public function customer_create()
    {

        if (request()->ctype == "cooperate") {
            $view = "customers.create_cooperate_customer";
        } else {
            $view = "customers.create_customer";
        }
        return view($view)->with('savingsprods', SavingsProduct::all())
            ->with('exrate', Exchangerate::all())
            ->with('officers', Accountofficer::all());
    }

    public function customer_show($id)
    {
        $balance = Saving::select('account_balance')->where('customer_id', $id)->first();
        return view('customers.show_customer')->with('cutoms', Customer::findorfail($id))
            ->with('balance', $balance);
    }

    public function customer_store(Request $request)
    {
        // dd($request->all());
        $this->logInfo("creating customer via core banking", $request->all());
        $request->validate([
            'title' => ['required', 'string'],
            'last_name' => ['required', 'string'],
            'first_name' => ['required', 'string'],
            'middle_name' => ['nullable', 'string'],
            'email' => ['required', 'email', 'unique:customers,email'],
            'marital_status' => ['required', 'string'],
            'phone' => ['required', 'digits:11', 'unique:customers,phone'],
            'dob' => ['required', 'date'],
            'account_type' => ['required', 'string'],
            'gender' => ['required', 'string'],
            'address' => ['required', 'string'],
            'country' => ['required', 'string'],
            'state' => ['required', 'string'],
            'lga' => ['required', 'string'],
            'religion' => ['nullable', 'string'],
            'bvn' => ['nullable', 'digits:11', 'unique:customers,bvn'],
            'occupation' => ['nullable', 'string'],
            'working_status' => ['nullable', 'string'],
            'means_of_id' => ['nullable', 'string'],
            'domicilary' => ['nullable', 'string'],
            'kin' => ['nullable', 'string'],
            'kin_address' => ['nullable', 'string'],
            'kin_phone' => ['nullable', 'digits:11'],
            'kin_relate' => ['nullable', 'string'],
            'account_number' => ['nullable', 'digits:10'],
            'refacct' => ['nullable', 'string'],
            'branchid' => ['required', 'integer'],
            'uredirect' => ['nullable', 'url'],
            'type' => ['required', 'string'],
            'account_officer' => ['required', 'string'],
        ]);




        $settings = new Setting();
        $branch = session()->has('branchid') ? session()->get('branchid')['bid'] : null;
        if (!empty($request->email) && ($request->tppe ?? '') !== "cooperate") {
            if (Customer::where('email', $request->email)->exists()) {
                return redirect($request->uredirect)->with('error', 'Account with this email already exists');
            }
        }

        if (($request->account_category ?? '') != '1') {
            if (Customer::where('acctno', $request->account_number)->exists()) {
                return redirect($request->uredirect)->with('error', 'Account number already exists');
            }
        }

        if (($request->type ?? '') !== "cooperate") {
            if (Customer::where('phone', $request->phone)->exists()) {
                return redirect($request->uredirect)->with('error', 'Account with this phone number already exists');
            }
        }

        if (!empty($request->bvn)) {
            if (strlen($request->bvn) != 11 || !ctype_digit($request->bvn)) {
                return redirect($request->uredirect)->with('error', 'Invalid BVN');
            }
        }

        if (!ctype_digit($request->account_number) || strlen($request->account_number) !== 10) {
            return redirect($request->uredirect)->with('error', 'Account number must be 10 digits');
        }
        $account_number = $request->account_number;

        $referralCode = strtolower(Str::random(6));
        $trnxpin = Hash::make((string) mt_rand(1111, 9999));
        $plainPassword = (string) mt_rand(11111111, 99999999);
        // $username = trim($request->first_name . ' ' . $request->last_name);
        $username = $this->generateUniqueUsername($request->first_name, $request->last_name);

        $customerData = [
            'user_id' => Auth::id(),
            'branch_id' => $request->branchid ?? null,
            'accountofficer_id' => $request->account_officer ?? null,
            'title' => $request->title ?? null,
            'first_name' => isset($request->first_name) ? strtolower($request->first_name) : null,
            'last_name' => isset($request->last_name) ? strtolower($request->last_name) : null,
            'middle_name' => $request->middle_name,
            'email' => $request->email ?? null,
            'phone' => $request->phone ?? null,
            'gender' => isset($request->gender) ? strtolower($request->gender) : null,
            'religion' => $request->religion ?? null,
            'marital_status' => $request->marital_status ?? null,
            'residential_address' => $request->address ?? null,
            'dob' => $request->dob ?? null,
            'username' => $username,
            'country' => $request->country ?? null,
            'state' => $request->state ?? null,
            'state_lga' => $request->lga ?? null,
            'account_type' => $request->account_type ?? null,
            'account_category' => $request->account_category ?? null,
            'exchangerate_id' => !empty($request->domicilary) ? $request->domicilary : null,
            'acctno' => $account_number,
            'refacct' => $request->refacct ?? null,
            'bvn' => $request->bvn ?? null,
            'next_kin' => $request->kin ?? null,
            'kin_address' => $request->kin_address ?? null,
            'kin_phone' => $request->kin_phone ?? null,
            'kin_relate' => $request->kin_relate ?? null,
            'occupation' => $request->occupation ?? null,
            'business_name' => isset($request->business_name) ? strtolower($request->business_name) : null,
            'working_status' => $request->working_status ?? null,
            'means_of_id' => $request->means_of_id ?? null,
            'transfer_limit' => '50000',
            'online_transfer_limit' => '50000',
            'password' => Hash::make($plainPassword),
            'phone_verify' => '0',
            'pin' => $trnxpin,
            'referral_code' => $referralCode,
            'reg_date' => now(),
            'source' => 'admin',
            'status' => '7',
            'ctype' => $request->type ?? null,
            'referee_name' => $request->ref_name ?? null,
            'referee_phone' => $request->ref_phone ?? null,
            'referee_bank' => $request->ref_bank ?? null,
            'referee_account_no' => $request->ref_account_number ?? null,
            'refferre_name' => $request->reff_name ?? null,
            'refferre_phone' => $request->reff_phone ?? null,
            'refferre_bank' => $request->reff_bank ?? null,
            'refferre_account_no' => $request->reff_account_number ?? null,
            'cac_no' => $request->cac_no ?? null,
            'incorporated_date' => $request->incorporatde_date ?? null,
            'tax_id' => $request->tax_id ?? null,
        ];

        // return $customerData;

        DB::beginTransaction();
        try {
            $customer = Customer::create($customerData);
            if ($request->hasFile('upload_id')) {
                $path = $request->file('upload_id')->store('uploads', 'public');
                $customer->update(['id_document' => 'storage/' . $path]);
            }

            if ($request->hasFile('photo')) {
                $path = $request->file('photo')->store('uploads', 'public');
                $customer->update(['photo' => 'storage/' . $path]);
            }

            if ($request->hasFile('signature')) {
                $path = $request->file('signature')->store('uploads', 'public');
                $customer->update(['signature' => 'storage/' . $path]);
            }

            $this->logInfo("customer created successfully", $customerData);

            $this->create_account(Auth::id(), $customer->id, $request->account_category);

            $name = ($request->type === "cooperate") ? ucwords($request->business_name) : ucwords($request->last_name . " " . $request->first_name);
            $msg = "Welcome {$name}, below is account details <br> Your Account Number: {$account_number}<br> Username: {$username} <br> Bank: " . ucwords($settings->getsettingskey('company_name')) . "<br><br> Thank you for choosing " . ucwords($settings->getsettingskey('company_name'));

            if (!empty($request->email)) {
                Email::create([
                    'user_id' => Auth::id(),
                    'subject' => 'Welcome to ' . ucwords($settings->getsettingskey('company_name')),
                    'message' => $msg,
                    'recipient' => $request->email,
                ]);

                Notification::route('mail', $request->email)
                    ->notify(new AppNotification('Account Registration', $msg));
                // $this->sendMail($msg, 'Account Registration', $request->email, 'Welcome to ' . ucwords($settings->getsettingskey('company_name')));
            }

            $usern = Auth::user()->last_name . " " . Auth::user()->first_name;
            $this->tracktrails(Auth::id(), $branch, $usern, 'customer', 'created a new customer account');

            DB::commit();

            $this->matrixPay->createAccount($customer);
            return redirect()->route('customer.pending')->with('success', 'Account created successfully');
        } catch (\Throwable $e) {
            DB::rollBack();
            $this->logInfo('customer create failed', ['error' => $e->getMessage()]);
            return redirect($request->uredirect)->with('error', 'Failed to create customer: ' . $e->getMessage());
        }
    }


    public function customer_edit($id)
    {

        if (request()->ctype == "cooperate") {
            $views = 'customers.edit_cooperate_customer';
        } else {
            $views = 'customers.edit_customer';
        }

        return view($views)->with('savingsprods', SavingsProduct::all())
            ->with('ced', Customer::findorfail($id))
            ->with('statuses', Status::all())
            ->with('exrate', Exchangerate::all())
            ->with('officers', Accountofficer::all());
    }

    public function customer_update(Request $r, $id)
    {

        $this->logInfo("customer updated by " . Auth::user()->first_name, $r->all());

        $validator = Validator::make($r->all(), [
            'last_name' => 'required|string',
            'first_name' => 'required|string',
            'middle_name' => 'required|string',
            'username' => [
                'required',
                Rule::unique('customers', 'username')->ignore($id),
            ],
            'account_type' => 'required|string',
            'account_category' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $getsetvalue = new Setting();

        $cusid = Customer::findorfail($id);

        $branch = session()->has('branchid') ? session()->get('branchid')['bid'] : null;

        if ($r->hasFile('upload_id')) {
            if (file_exists($cusid->upload_id)) {
                unlink($cusid->upload_id);
            }
            $photoid = $r->file('upload_id');
            $newphotoid = time() . "_" . $photoid->getClientOriginalName();
            $photoid->move('uploads', $newphotoid);
            $cusid->upload_id = 'uploads/' . $newphotoid;
        }

        if ($r->hasFile('photo')) {
            if (file_exists($cusid->photo)) {
                unlink($cusid->photo);
            }
            $photp = $r->file('photo');
            $newphoto = time() . "_" . $photp->getClientOriginalName();
            $photp->move('uploads', $newphoto);
            $cusid->photo = 'uploads/' . $newphoto;
        }

        if ($r->hasFile('signature')) {
            if (file_exists($cusid->signature)) {
                unlink($cusid->signature);
            }
            $signature = $r->file('signature');
            $newsignature = time() . "_" . $signature->getClientOriginalName();
            $signature->move('uploads', $newsignature);
            $cusid->signature = 'uploads/' . $newsignature;
        }

        $cusid->accountofficer_id = $r->account_officer;
        $cusid->title = $r->title;
        $cusid->first_name = $r->first_name;
        $cusid->last_name = $r->last_name;
        $cusid->middle_name = $r->middle_name;

        $cusid->email = $r->email;
        $cusid->phone = $r->phone;
        $cusid->gender = $r->gender;
        $cusid->religion = $r->religion;
        $cusid->marital_status = $r->marital_status;
        $cusid->residential_address = $r->address;
        $cusid->username = $r->username;
        $cusid->dob = $r->dob;
        $cusid->country = $r->country;
        $cusid->state = $r->state;
        $cusid->state_lga = $r->lga;
        $cusid->account_type = $r->account_type;
        $cusid->account_category = $r->account_category;
        $cusid->exchangerate_id = !empty($r->domicilary) ? $r->domicilary : null;
        $cusid->refacct = $r->refacct;
        $cusid->bvn = $r->bvn;
        $cusid->next_kin = $r->kin;
        $cusid->kin_address = $r->kin_address;
        $cusid->kin_phone = $r->kin_phone;
        $cusid->kin_relate = $r->kin_relate;
        $cusid->transfer_limit = $r->transfer_limit;
        $cusid->online_transfer_limit = $r->online_transfer_limit;
        $cusid->lien = $r->lien_account;
        $cusid->lien_status = $r->lien_status;
        $cusid->lien_amount = $r->lien_amount;
        $cusid->occupation = $r->occupation;
        $cusid->business_name = $r->business_name;
        $cusid->working_status = $r->working_status;
        $cusid->means_of_id = $r->means_of_id;
        $cusid->phone_verify = $r->phone_status;
        $cusid->status = $r->account_status;
        $cusid->whatsapp = $r->whatsapp;
        $cusid->ussd = null;
        $cusid->enable_email_alert = $r->enable_email_alert;
        $cusid->enable_sms_alert = $r->enable_sms_alert;
        $cusid->internet_banking = $r->enable_internet_banking;
        $cusid->referee_name = $r->ref_name;
        $cusid->referee_phone = $r->ref_phone;
        $cusid->referee_bank = $r->ref_bank;
        $cusid->referee_account_no = $r->ref_account_number;
        $cusid->refferre_name = $r->reff_name;
        $cusid->refferre_phone = $r->reff_phone;
        $cusid->refferre_bank = $r->reff_bank;
        $cusid->refferre_account_no = $r->reff_account_number;
        $cusid->cac_no = $r->cac_no;
        $cusid->incorporated_date = $r->incorporatde_date;
        $cusid->tax_id = $r->tax_id;

        if ($r->account_status == "1") {

            $cusid->failed_balance = null;
            $cusid->failed_logins = null;
            $cusid->failed_pin = null;
        }

        $cusid->save();

        //adding to savings
        $this->create_account(Auth::user()->id, $id, $r->account_category);

        $usern = Auth::user()->last_name . " " . Auth::user()->first_name;
        $this->tracktrails(Auth::user()->id, $branch, $usern, 'customer', 'updated a customer account');

        // if($getsetvalue->getsettingskey('enable_virtual_ac') == '1'){
        // $response= Http::withHeaders([
        //                "PublicKey" => env('PUBLIC_KEY'),
        //                "EncryptKey" => env('ENCRYPT_KEY'),
        //                "Content-Type" => "application/json",
        //                "Accept" => "application/json"
        //            ])->post(env('VIRTUAL_ACCOUNT_URL'),[
        //             "settlement_accountno" => env('SETTLEMENT_ACCOUNT'),
        //                 "account_name" =>  $r->last_name." ".$r->first_name,
        //                 "accountno" => $cusid->acctno
        //            ]);
        // }

        return redirect(route('customer.view', ['id' => $id]) . "?ctype=" . $cusid->ctype)->with('success', 'Record Updated');
    }

    public function customer_delete($id)
    {
        $cust = Customer::findorfail($id);
        if (file_exists($cust->photo)) {
            unlink($cust->photo);
        }
        if (file_exists($cust->signature)) {
            unlink($cust->signature);
        }
        $cust->delete();
        $branch = session()->has('branchid') ? session()->get('branchid')['bid'] : null;

        $usern = Auth::user()->last_name . " " . Auth::user()->first_name;
        $this->tracktrails(Auth::user()->id, $branch, $usern, 'customer', 'deleted a customer account');

        return redirect()->back()->with('success', 'Record Deleted');
    }

    //pin / password reset
    public function customer_reset_pin_password(Request $r)
    {

        $branch = session()->has('branchid') ? session()->get('branchid')['bid'] : null;

        $getsetvalue = new Setting();
        $customerAccunt = Customer::where('id', $r->userid)->first();

        if ($r->type == "Send Password Reset") {
            $this->logInfo("Password reset by " . Auth::user()->first_name, "");

            $passwrd = strtolower(Str::random(8));
            $customerAccunt->password = Hash::make($passwrd);
            $customerAccunt->failed_logins = NUll;
            $customerAccunt->save();


            $usern = Auth::user()->last_name . " " . Auth::user()->first_name;
            $this->tracktrails(Auth::user()->id, $branch, $usern, 'customer', 'sent password reset to customer account');

            $name = !is_null($customerAccunt->business_name) ? ucwords($customerAccunt->business_name) : $customerAccunt->last_name . " " . $customerAccunt->first_name;

            $msg = "Hi " . $name . "<br><br> Below is your Password <br> " . $passwrd . " <br><br>Kindly upload your Valid ID and link BVN.";
            Email::create([
                'user_id' => Auth::user()->id,
                'subject' => 'Password Reset Request from ' . ucwords($getsetvalue->getsettingskey('company_name')),
                'message' => $msg,
                'recipient' => $customerAccunt->email,
            ]);

            $this->sendMail($msg, 'Password Reset Request', $customerAccunt->email, 'Password Reset Request from ' . ucwords($getsetvalue->getsettingskey('company_name')));

            // Mail::send(['html' => 'mails.sendmail'],[
            //     'msg' => $msg,
            //      'type' => 'Password Reset Request'
            // ],function($mail)use($customerAccunt,$getsetvalue){
            //     $mail->from($getsetvalue->getsettingskey('company_email'),ucwords($getsetvalue->getsettingskey('company_name')));
            //     $mail->to($customerAccunt->email);
            //     $mail->subject('Password Reset Request from '.ucwords($getsetvalue->getsettingskey('company_name')));
            // });

            //'customer.edit',['id' => $r->userid]
        } elseif ($r->type == "Send Pin Reset") {

            $this->logInfo("Pin reset by " . Auth::user()->first_name, "");

            $pin = mt_rand('1111', '9999');
            $customerAccunt->pin = Hash::make($pin);
            $customerAccunt->failed_pin = null;
            $customerAccunt->save();


            $usern = Auth::user()->last_name . " " . Auth::user()->first_name;
            $this->tracktrails(Auth::user()->id, $branch, $usern, 'customer', 'sent pin reset customer account');

            $name = !is_null($customerAccunt->business_name) ? ucwords($customerAccunt->business_name) : $customerAccunt->last_name . " " . $customerAccunt->first_name;

            $msg = "Hi " . $name . "<br><br> Below is your Pin <br> " . $pin . " <br><br>Kindly upload your Valid ID and link BVN.";
            Email::create([
                'user_id' => Auth::user()->id,
                'subject' => 'Pin Reset Request from ' . ucwords($getsetvalue->getsettingskey('company_name')),
                'message' => $msg,
                'recipient' => $customerAccunt->email,
            ]);

            $this->sendMail($msg, 'Pin Reset Request', $customerAccunt->email, 'Pin Reset Request from ' . ucwords($getsetvalue->getsettingskey('company_name')));

            // Mail::send(['html' => 'mails.sendmail'],[
            //     'msg' => $msg,
            //      'type' => 'Pin Reset Request'
            // ],function($mail)use($customerAccunt,$getsetvalue){
            //     $mail->from($getsetvalue->getsettingskey('company_email'),ucwords($getsetvalue->getsettingskey('company_name')));
            //     $mail->to($customerAccunt->email);
            //     $mail->subject('Pin Reset Request from '.ucwords($getsetvalue->getsettingskey('company_name')));
            // });

            //return redirect()->back()->with('success','Pin Reset Sent successfully');
        }

        return redirect()->back()->with('success', 'Reset Request Sent successfully');
    }
    //activate customers account
    public function activate_close_customer(Request $r)
    {
        if (!empty($r->customerid)) {
            if ($r->cmdupdatestatus == "Activate Account(s)") {
                foreach ($r->customerid as $customerid) {
                    Customer::where('id', $customerid)->update([
                        'status' => "1",
                        'failed_balance' => null,
                        'failed_logins' => null,
                        'failed_pin' => null
                    ]);
                }
                $branch = session()->has('branchid') ? session()->get('branchid')['bid'] : null;

                $usern = Auth::user()->last_name . " " . Auth::user()->first_name;
                $this->tracktrails(Auth::user()->id, $branch, $usern, 'customer', 'activated a customer account');

                return redirect()->back()->with('success', 'Account Activated');
            } elseif ($r->cmdupdatestatus == "Close Account(s)") {
                foreach ($r->customerid as $customerid) {
                    Customer::where('id', $customerid)->update([
                        'status' => "2"
                    ]);
                }
                $branch = session()->has('branchid') ? session()->get('branchid')['bid'] : null;

                $usern = Auth::user()->last_name . " " . Auth::user()->first_name;
                $this->tracktrails(Auth::user()->id, $branch, $usern, 'customer', 'deactivated/closed a customer account');

                return redirect()->back()->with('success', 'Account Closed');
            }
        } else {
            return redirect()->back();
        }
    }

    public function customer_activate($id)
    {
        $this->logInfo("customer activated by" . Auth::user()->first_name, "");

        $branch = session()->has('branchid') ? session()->get('branchid')['bid'] : null;

        Customer::where('id', $id)->update([
            'status' => request()->status,
            'failed_balance' => null,
            'failed_logins' => null,
            'failed_pin' => null
        ]);

        $usern = Auth::user()->last_name . " " . Auth::user()->first_name;
        $this->tracktrails(Auth::user()->id, $branch, $usern, 'customer', 'activated a customer account');

        return redirect()->back()->with('success', 'Account Activated');
    }

    public function customer_closed($id)
    {
        $this->logInfo("customer account closed by" . Auth::user()->first_name, "");

        Customer::where('id', $id)->update([
            'status' => request()->status
        ]);
        $branch = session()->has('branchid') ? session()->get('branchid')['bid'] : null;

        $usern = Auth::user()->last_name . " " . Auth::user()->first_name;
        $this->tracktrails(Auth::user()->id, $branch, $usern, 'customer', 'deactivated/closed a customer account');

        return redirect()->back()->with('success', 'Account Closed');
    }

    public function create_mail($id = null)
    {
        $getemail = "";
        if (!is_null($id) && request()->sendmail == true) {
            $getemail = Customer::select('email')->where('id', $id)->first();
        }
        return view('communicate.create_emails')->with('remail', $getemail->email)
            ->with('cusem', Customer::select('email')->where('status', '1')->get());
    }

    public function create_sms($id = null)
    {
        $getemail = "";
        if (!is_null($id) && request()->sendsms == true) {
            $cusromers = Customer::select('first_name', 'last_name', 'phone')->where('id', $id)->first();

            return view('communicate.create_sms')->with('cusms', $cusromers);
        } else {
            return view('communicate.create_sms')->with('cusms', Customer::select('first_name', 'last_name', 'phone')->where('status', '1')->get());
        }
    }

    public function getpendingcust()
    {
        if (Auth::user()->roles()->first()->name == 'account officer') {
            $acofficer = Accountofficer::where('user_id', Auth::user()->id)->first();
            $pending = Customer::where('accountofficer_id', $acofficer->id)
                ->where('status', '7')
                ->count();
            return $pending;
        } else {
            $pending = Customer::where('status', '7')->count();
            return $pending;
        }
    }

    public function getclosecust()
    {
        if (Auth::user()->roles()->first()->name == 'account officer') {
            $acofficer = Accountofficer::where('user_id', Auth::user()->id)->first();
            $closeing = Customer::where('accountofficer_id', $acofficer->id)
                ->where('status', '2')
                ->count();
            return $closeing;
        } else {
            $closeing = Customer::where('status', '2')->count();
            return $closeing;
        }
    }

    public function getrestricust()
    {
        if (Auth::user()->roles()->first()->name == 'account officer') {
            $acofficer = Accountofficer::where('user_id', Auth::user()->id)->first();
            $closeing = Customer::where('accountofficer_id', $acofficer->id)
                ->whereBetween('status', ['4', '6'])
                ->count();
            return $closeing;
        } else {
            $closeing = Customer::whereBetween('status', ['4', '6'])->count();
            return $closeing;
        }
    }
    public function getdomcust()
    {
        if (Auth::user()->roles()->first()->name == 'account officer') {
            $acofficer = Accountofficer::where('user_id', Auth::user()->id)->first();
            $closeing = Customer::where('accountofficer_id', $acofficer->id)
                ->where('status', '8')
                ->count();
            return $closeing;
        } else {
            $closeing = Customer::where('status', '8')->count();
            return $closeing;
        }
    }

    public function getactivecust()
    {
        if (Auth::user()->roles()->first()->name == 'account officer') {
            $acofficer = Accountofficer::where('user_id', Auth::user()->id)->first();
            $activeg = Customer::where('accountofficer_id', $acofficer->id)
                ->where('status', '1')
                ->get()->count();
            return $activeg;
        } else {
            $activeg = Customer::where('status', '1')->count();
            return $activeg;
        }
    }




       public function store_upload_customer(Request $r)
    {
        $this->validate($r, [
            'customer_file' => ['required', 'mimes:csv,txt', 'max:10240']
        ]);

        $csvfile = $r->file('customer_file');
        $newcsvfile = time() . "_" . preg_replace(
            '/[^A-Za-z0-9\-\_\.]/',
            '_',
            $csvfile->getClientOriginalName()
        );

        $csvfile->move(public_path('uploads'), $newcsvfile);
        $filepath = public_path('uploads/' . $newcsvfile);

        $rows = [];

        if (($file = fopen($filepath, 'r')) !== false) {
            $header = fgetcsv($file);

            while (($data = fgetcsv($file)) !== false) {
                $row = [];
                foreach ($header as $i => $colName) {
                    $key = strtolower(str_replace(
                        ' ',
                        '_',
                        trim(preg_replace('/^\xEF\xBB\xBF/', '', $colName))
                    ));
                    $row[$key] = $data[$i] ?? '';
                }
                $rows[] = $row;
            }
            fclose($file);
        }

        $summary = $this->upload_customers_via_excel($rows);
        @unlink($filepath);

        $messages = [];
        $messages[] = "{$summary['total_rows']} row(s) processed";

        if ($summary['inserted'] > 0) {
            $messages[] = "{$summary['inserted']} customer(s) added";
        }
        if ($summary['duplicates'] > 0) {
            $messages[] = "{$summary['duplicates']} duplicate record(s) skipped";
        }
        if ($summary['skipped'] > 0) {
            $messages[] = "{$summary['skipped']} invalid row(s) skipped";
        }
        if ($summary['failed'] > 0) {
            $messages[] = "{$summary['failed']} row(s) failed";
        }

        $alertType = $summary['inserted'] > 0 ? 'success' : 'warning';

        return redirect()
            ->back()
            ->with('success', implode('. ', $messages))
            ->with('csv_status', $alertType)
            ->with('csv_summary', $summary);
    }

    /* =====================================================
       CORE IMPORT LOGIC
    ===================================================== */
    public function upload_customers_via_excel(array $rows): array
    {
        $user = Auth::user();

        $summary = [
            'total_rows' => count($rows),
            'inserted'   => 0,
            'duplicates' => 0,
            'skipped'    => 0,
            'failed'     => 0,
        ];

        Log::info('CSV IMPORT STARTED', ['rows' => count($rows)]);

        foreach ($rows as $i => $row) {

            $rowNo = $i + 2;

            /* ======================
               BASIC VALIDATION
            ====================== */
            $fname = trim($row['first_name'] ?? '');
            $lname = trim($row['last_name'] ?? '');
            $phone = preg_replace('/\D/', '', $row['phone'] ?? '');
            $email = strtolower(trim($row['email'] ?? ''));
            $bvn   = preg_replace('/\D/', '', $row['bvn'] ?? '');

            if ($fname === '' || $lname === '' || strlen($phone) < 10) {
                $summary['skipped']++;
                continue;
            }

            /* ======================
               ACCOUNT CATEGORY
            ====================== */
            $categoryRaw = strtolower(trim($row['account_category'] ?? ''));

            if (str_contains($categoryRaw, 'savings')) {
                $acctPrefix = 1;
                $ctype = 'savings';
            } elseif (str_contains($categoryRaw, 'current')) {
                $acctPrefix = 2;
                $ctype = 'current';
            } else {
                $summary['skipped']++;
                continue;
            }

            /* ======================
               GENERATE UNIQUE DATA
            ====================== */
            $acctno   = $this->generateUniqueAccountNumber($acctPrefix);
            $username = $this->generateUniqueUsername($fname, $lname);

            /* ======================
               DUPLICATE CHECK
            ====================== */
            if (
                ($email && Customer::where('email', $email)->exists()) ||
                ($phone && Customer::where('phone', $phone)->exists()) ||
                ($bvn && Customer::where('bvn', $bvn)->exists()) ||
                Customer::where('username', $username)->exists()
            ) {
                $summary['duplicates']++;
                continue;
            }

            /* ======================
               DOB
            ====================== */
            $dob = null;
            if (!empty($row['dob'])) {
                try {
                    $dob = Carbon::parse($row['dob'])->format('Y-m-d');
                } catch (\Throwable $e) {
                    // ignore invalid DOB
                }
            }

            /* ======================
               CUSTOMER PAYLOAD
            ====================== */
            $payload = [
                'user_id'    => $user->id,
                'branch_id'  => $user->branch_id,
                'title'      => $row['title'] ?? null,
                'first_name' => $fname,
                'last_name'  => $lname,
                'email'      => $email ?: null,
                'phone'      => $phone,
                'bvn'        => $bvn ?: null,
                'acctno'     => $acctno,
                'username'   => $username,
                'dob'        => $dob,
                'ctype'      => $ctype,
                'status'     => 7,
                'reg_date'   => now(),
                'password'   => Hash::make(rand(11111111, 99999999)),
                'pin'        => Hash::make(rand(1111, 9999)),
                'source'     => 'admin',
            ];

            /* ======================
               INSERT WITH TRANSACTION
            ====================== */
            DB::beginTransaction();

            try {
                $customer = Customer::create($payload);

                /* ======================
                   CREATE SAVINGS ACCOUNT
                ====================== */
                if ($ctype === 'savings') {
                    Saving::create([
                        'user_id'            => $user->id,
                        'customer_id'        => $customer->id,
                        'savings_product_id' => 1, // default savings product
                        'account_balance'    => 0.00,
                        'ledger_balance'     => 0.00,
                        'lien_amount'        => 0.00,
                        'lien_deducted'      => 0.00,
                        'lien_remaining'     => 0.00,
                        'branch_id'          => $user->branch_id,
                    ]);
                }

                DB::commit();

                $summary['inserted']++;

            } catch (\Throwable $e) {
                DB::rollBack();

                $summary['failed']++;

                Log::error("IMPORT FAILED ROW {$rowNo}", [
                    'error' => $e->getMessage()
                ]);
            }
        }

        Log::info('CSV IMPORT FINISHED', $summary);

        return $summary;
    }






    public function export_customerbalance_data()
    {
        $filter = request()->filter == true ? true : false;
        $searchval = !empty(request()->searchval) ? request()->searchval : null;
        return Excel::download(new CustomersBalanceExport($searchval, $filter), 'Customer_balance.xlsx');
    }
}
