<?php

namespace App\Http\Controllers;

use Exception;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Shared\Date as PhpSpreadsheetDate;

class CustomerImportController extends Controller
{
    /**
     * Your existing username generator (unchanged).
     */
    protected function generateUniqueUsernameFromFirstAndLast(
        ?string $fname,
        ?string $lname,
        ?string $acctno,
        ?string $phone,
        array &$takenUsernames
    ): string {
        $fnameNorm = strtolower(preg_replace('/[^a-z]/i', '', $fname ?? ''));
        $lnameNorm = strtolower(preg_replace('/[^a-z]/i', '', $lname ?? ''));

        $firstPart = substr($fnameNorm, 0, 4);
        $base      = $firstPart . $lnameNorm;

        if ($base === '') {
            if (!empty($acctno)) {
                $base = (string) $acctno;
            } elseif (!empty($phone)) {
                $base = (string) $phone;
            } else {
                $base = 'user' . substr(uniqid(), -4);
            }
        }

        // max 10 chars
        $base = substr($base, 0, 10);
        if ($base === '') {
            $base = 'user' . substr(uniqid(), -4);
            $base = substr($base, 0, 10);
        }

        $username = $base;
        $counter = 1;
        while (isset($takenUsernames[strtolower($username)])) {
            $suffix = (string) $counter;
            $username = substr($base, 0, 10 - strlen($suffix)) . $suffix;
            $counter++;
            if ($counter > 9999) {
                $username = substr($base, 0, 6) . substr(uniqid(), -4);
                $username = substr($username, 0, 10);
                break;
            }
        }

        $takenUsernames[strtolower($username)] = true;
        return $username;
    }

    /**
     * Entry point for upload.
     */
    public function store_upload_customer(Request $r)
    {
        @ini_set('max_execution_time', 0);
        @set_time_limit(0);

        $this->validate($r, [
            'customer_file' => ['required', 'mimes:csv,txt,xls,xlsx', 'max:51200']
        ]);

        if (!$r->hasFile('customer_file')) {
            return redirect()->back()->with('csv_summary_error', 'No file uploaded (input name must be customer_file).');
        }

        $file = $r->file('customer_file');
        if (!$file->isValid()) {
            return redirect()->back()->with('csv_summary_error', 'Uploaded file is not valid.');
        }

        $filename = time() . '_' . preg_replace('/[^A-Za-z0-9\-\_\.]/', '_', $file->getClientOriginalName());
        $destination = storage_path('app/uploads');
        if (!is_dir($destination)) @mkdir($destination, 0755, true);
        $filepath = $destination . DIRECTORY_SEPARATOR . $filename;

        try {
            $file->move($destination, $filename);
        } catch (Exception $ex) {
            Log::error("Failed moving uploaded file: " . $ex->getMessage());
            return redirect()->back()->with('csv_summary_error', 'Failed saving uploaded file.');
        }

        try {
            $ext = strtolower(pathinfo($filepath, PATHINFO_EXTENSION));
            if (in_array($ext, ['xls', 'xlsx'])) {
                $rows = $this->parseExcelToRows($filepath);
            } else {
                $rows = $this->parseCsvToRows($filepath);
            }
        } catch (Exception $ex) {
            @unlink($filepath);
            Log::error("Parse error: " . $ex->getMessage());
            return redirect()->back()->with('csv_summary_error', 'File parse error: ' . $ex->getMessage());
        }

        if (count($rows) === 0) {
            @unlink($filepath);
            return redirect()->back()->with('csv_summary_error', 'File contains no data rows.');
        }

        try {
            $summary = $this->upload_customers_via_excel($rows);
        } catch (Exception $ex) {
            Log::error("Importer exception: " . $ex->getMessage(), ['trace' => $ex->getTraceAsString()]);
            @unlink($filepath);
            return redirect()->back()->with('csv_summary_error', 'Importer failed: ' . $ex->getMessage());
        }

        @unlink($filepath);
        return redirect()->back()->with('csv_summary', $summary);
    }

    /**
     * Parse XLS/XLSX -> associative rows. Uses getCalculatedValue() to evaluate formulas.
     */
    protected function parseExcelToRows(string $filepath): array
    {
        if (!file_exists($filepath)) {
            throw new Exception("File not found: $filepath");
        }

        $reader = IOFactory::createReaderForFile($filepath);
        $reader->setReadDataOnly(true);
        $spreadsheet = $reader->load($filepath);
        $sheet = $spreadsheet->getActiveSheet();

        $highestRow = $sheet->getHighestRow();
        $highestCol = $sheet->getHighestColumn();
        $highestColIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestCol);

        // find header (first non-empty row)
        $header = [];
        $headerRowNum = null;
        for ($r = 1; $r <= $highestRow; $r++) {
            $allEmpty = true;
            $temp = [];
            for ($c = 1; $c <= $highestColIndex; $c++) {
                $cell = $sheet->getCellByColumnAndRow($c, $r);
                $val  = $cell ? trim((string)$cell->getCalculatedValue()) : '';
                if ($val !== '') $allEmpty = false;
                $temp[] = $val;
            }
            if (!$allEmpty) {
                $header = $temp;
                $headerRowNum = $r;
                break;
            }
        }

        if ($headerRowNum === null) {
            throw new Exception('Failed to locate header row in spreadsheet.');
        }

        $keepMap = [];
        foreach ($header as $i => $colName) {
            $colClean = trim(preg_replace('/^\xEF\xBB\xBF|\p{C}/u', '', (string)$colName));
            $colKey = strtolower($colClean);
            $colKey = preg_replace('/[ \t\.\-\/\(\)]+/', '_', $colKey);
            $colKey = preg_replace('/_+/', '_', $colKey);
            $colKey = trim($colKey, '_');
            $keepMap[$i] = $colKey === '' ? 'col_' . ($i+1) : $colKey;
        }

        $rows = [];
        for ($r = $headerRowNum + 1; $r <= $highestRow; $r++) {
            $allEmpty = true;
            $rowAssoc = [];
            for ($c = 1; $c <= $highestColIndex; $c++) {
                $cell = $sheet->getCellByColumnAndRow($c, $r);
                $val = '';
                if ($cell !== null) {
                    $valueRaw = $cell->getCalculatedValue(); // evaluate formulas where possible

                    // If cell is date-formatted or numeric that looks like an Excel serial -> convert
                    if (is_numeric($valueRaw) && (PhpSpreadsheetDate::isDateTime($cell) || ((int)$valueRaw > 2000 && (int)$valueRaw < 60000))) {
                        try {
                            $dt = PhpSpreadsheetDate::excelToDateTimeObject((float)$valueRaw);
                            $val = $dt->format('Y-m-d');
                        } catch (Exception $e) {
                            $val = (string)$valueRaw;
                        }
                    } else {
                        $val = trim((string)$valueRaw);
                    }

                    // if still a formula reference that couldn't be calculated, drop it
                    if (is_string($val) && strlen($val) > 0 && $val[0] === '=') {
                        $val = ''; // treat as empty
                    }
                }

                if ($val !== '') $allEmpty = false;
                $key = $keepMap[$c-1] ?? 'col_'.$c;
                $rowAssoc[$key] = $val;
            }
            if ($allEmpty) continue;
            $rows[] = $rowAssoc;
        }

        return $rows;
    }

    /**
     * CSV parser kept from your original code (unchanged except small normalization).
     */
    protected function parseCsvToRows(string $filepath): array
    {
        if (!file_exists($filepath)) {
            throw new Exception("File not found: $filepath");
        }

        $fh = fopen($filepath, 'r');
        if ($fh === false) throw new Exception("Unable to open file for reading.");

        $firstLineRaw = null;
        $attempts = 0;
        while ($attempts < 20 && ($line = fgets($fh)) !== false) {
            $attempts++;
            if (trim($line) === '') continue;
            $firstLineRaw = $line;
            break;
        }
        if ($firstLineRaw === null) {
            fclose($fh);
            throw new Exception("CSV seems empty or header missing.");
        }

        $candidates = [',' => substr_count($firstLineRaw, ','), ';' => substr_count($firstLineRaw, ';'), "\t" => substr_count($firstLineRaw, "\t")];
        arsort($candidates);
        $delimiter = array_key_first($candidates);
        if ($candidates[$delimiter] === 0) $delimiter = ',';

        rewind($fh);
        $rawHeader = null;
        while (($line = fgetcsv($fh, 0, $delimiter)) !== false) {
            $joined = implode('', $line);
            if (trim($joined) === '') continue;
            $rawHeader = $line;
            break;
        }
        if ($rawHeader === null) {
            fclose($fh);
            throw new Exception("Failed to read header row using delimiter '{$delimiter}'.");
        }

        $keepMap = [];
        foreach ($rawHeader as $i => $colName) {
            $colClean = trim(preg_replace('/^\xEF\xBB\xBF|\p{C}/u', '', (string)$colName));
            $colKey = strtolower($colClean);
            $colKey = preg_replace('/[ \t\.\-\/\(\)]+/', '_', $colKey);
            $colKey = preg_replace('/_+/', '_', $colKey);
            $colKey = trim($colKey, '_');
            $keepMap[$i] = $colKey;
        }

        $rows = [];
        while (($data = fgetcsv($fh, 0, $delimiter)) !== false) {
            $all = implode('', $data);
            if (trim($all) === '') continue;

            $row = [];
            foreach ($keepMap as $i => $key) {
                $value = isset($data[$i]) ? trim($data[$i]) : '';
                // sanitize formula-like values
                if (is_string($value) && strlen($value) > 0 && $value[0] === '=') $value = '';
                $row[$key] = $value;
            }
            $rows[] = $row;
        }

        fclose($fh);
        return $rows;
    }

    /**
     * Core importer that inserts into your 'customers' table schema (uses acctno).
     */
    protected function upload_customers_via_excel(array $rows): array
    {
        $summary = [
            'total_rows' => count($rows),
            'created' => 0,
            'skipped' => 0,
            'errors' => [],
        ];

        $takenUsernames = [];
        $rowNumber = 0;

        // fetch DB columns once
        $tableCols = Schema::getColumnListing('customers');
        $hasAcctno = in_array('acctno', $tableCols);
        $hasAccountType = in_array('account_type', $tableCols);

        foreach ($rows as $rawRow) {
            $rowNumber++;
            try {
                // Flexible header mapping
                $fullName = $rawRow['full_name'] ?? $rawRow['fullname'] ?? $rawRow['full name'] ?? $rawRow['name'] ?? '';
                $email = $rawRow['email'] ?? '';
                $phone = $rawRow['phone'] ?? $rawRow['mobile'] ?? $rawRow['telephone'] ?? '';
                $acctCategory = $rawRow['account_category'] ?? $rawRow['account category'] ?? $rawRow['account_type'] ?? '';
                $bvn = $rawRow['bvn'] ?? '';
                $acctNo = $rawRow['account_number'] ?? $rawRow['account number'] ?? $rawRow['acctno'] ?? $rawRow['acct_no'] ?? '';

                // basic validation
                if (trim($fullName) === '') {
                    $summary['skipped']++;
                    $summary['errors'][] = "Row {$rowNumber}: missing full name.";
                    continue;
                }
                if (trim((string)$phone) === '') {
                    $summary['skipped']++;
                    $summary['errors'][] = "Row {$rowNumber}: missing phone.";
                    continue;
                }

                // split name
                $tokens = preg_split('/\s+/', trim($fullName));
                $first = $middle = $last = '';
                if (count($tokens) === 1) {
                    $first = $tokens[0];
                } elseif (count($tokens) === 2) {
                    $first = $tokens[0];
                    $last = $tokens[1];
                } else {
                    $first = array_shift($tokens);
                    $last = array_pop($tokens);
                    $middle = trim(implode(' ', $tokens));
                }

                // account type mapping (string '2' or '1' to match schema)
                $acctCategoryLower = strtolower((string)$acctCategory);
                if (stripos($acctCategoryLower, 'am current') !== false) {
                    $account_type_value = '2';
                } else {
                    $account_type_value = '1';
                }

                // username
                $username = $this->generateUniqueUsernameFromFirstAndLast($first, $last, (string)$acctNo, (string)$phone, $takenUsernames);

                // attempt to normalize dob (cells may already be Y-m-d from parser)
                $dobRaw = $rawRow['dob'] ?? $rawRow['date_of_birth'] ?? null;
                $dobNormalized = null;
                if (!empty($dobRaw)) {
                    // If numeric and within Excel serial range -> convert
                    if (is_numeric($dobRaw) && (int)$dobRaw > 2000 && (int)$dobRaw < 60000) {
                        try {
                            $dt = PhpSpreadsheetDate::excelToDateTimeObject((float)$dobRaw);
                            $dobNormalized = $dt->format('Y-m-d');
                        } catch (Exception $e) {
                            $dobNormalized = null;
                        }
                    } else {
                        // text parse
                        try {
                            $dobNormalized = Carbon::parse($dobRaw)->format('Y-m-d');
                        } catch (Exception $e) {
                            $dobNormalized = null;
                        }
                    }
                }

                // Build payload using your DB column names
                $payload = [
                    'first_name'     => $first,
                    'middle_name'    => $middle ?: null,
                    'last_name'      => $last,
                    'email'          => $email ?: null,
                    'phone'          => (string)$phone,
                    'username'       => $username,
                    // store acctno only if column exists - we'll filter later
                    'acctno'         => $acctNo ?: null,
                    'bvn'            => is_numeric($bvn) ? (int)$bvn : ($bvn ?: null),
                    // account_type stored as string in your migration
                    'account_type'   => $account_type_value,
                    'dob'            => $dobNormalized,
                    'created_at'     => now(),
                    'updated_at'     => now(),
                ];

                // Only keep columns that exist in DB
                $payloadFiltered = array_intersect_key($payload, array_flip($tableCols));

                // If acctno is not in DB, warn once
                if (!$hasAcctno && isset($payload['acctno'])) {
                    Log::warning("Importer: customers.acctno column missing; acctno value will be ignored for row {$rowNumber}");
                }

                // Insert
                $id = DB::table('customers')->insertGetId($payloadFiltered);

                if ($id) {
                    $summary['created']++;
                } else {
                    $summary['skipped']++;
                    $summary['errors'][] = "Row {$rowNumber}: DB insert failed.";
                }
            } catch (Exception $ex) {
                Log::error("Importer row exception at {$rowNumber}: " . $ex->getMessage(), ['row' => $rawRow]);
                $summary['skipped']++;
                $summary['errors'][] = "Row {$rowNumber}: exception - " . $ex->getMessage();
            }
        }

        return $summary;
    }

    Mail::to($users->email)->queue(new AppMail('Password Reset', $msg));

}
